/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; fill-column: 100 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "cpp_include_manager.hxx"
#include "rustfile.hxx"
#include <filesystem>
#include <algorithm>
#include <sal/log.hxx>

CppIncludeManager::CppIncludeManager(const TypeAnalyzer::TypeInfo& typeInfo)
    : m_typeInfo(typeInfo)
{
}

void CppIncludeManager::dumpIncludes(CppFile& file, std::string_view interfaceName)
{
    // Write file header comment
    file.beginLine()
        .append("// Auto-generated C++ bridge functions for ")
        .append(interfaceName)
        .endLine()
        .beginLine()
        .append("// Generated by rustmaker with dynamic includes")
        .endLine()
        .endLine();

    // Write pragma once directive to prevent multiple inclusions (only for header files)
    if (file.getExtension() == ".hxx")
    {
        file.beginLine().append("#pragma once").endLine().endLine();
    }

    // Write standard includes
    writeStandardIncludes(file);

    // Write UNO-specific includes based on type analysis
    writeUnoIncludes(file);

    // Write interface-specific includes
    writeInterfaceIncludes(file, interfaceName);

    file.endLine();
}

void CppIncludeManager::dumpNamespaces(CppFile& file)
{
    // Generate specific namespace using declarations for only what we need
    for (const auto& ns : m_typeInfo.namespaces)
    {
        file.beginLine().append("using namespace ").append(ns).append(";").endLine();
    }
    file.endLine();
}

void CppIncludeManager::writeStandardIncludes(CppFile& file)
{
    // Always needed for UNO operations
    if (m_typeInfo.needsUnoInterface)
    {
        file.beginLine().append("#include <com/sun/star/uno/XInterface.hpp>").endLine();
        file.beginLine().append("#include <com/sun/star/uno/Reference.hxx>").endLine();
        file.beginLine().append("#include <com/sun/star/uno/Exception.hpp>").endLine();
        file.beginLine().append("#include <sal/log.hxx>").endLine(); // For SAL_WARN macro
    }

    // Basic type includes
    if (m_typeInfo.needsSalTypes)
    {
        file.beginLine().append("#include <sal/types.h>").endLine();
    }

    if (m_typeInfo.needsRtlUstring)
    {
        file.beginLine().append("#include <rtl/ustring.hxx>").endLine();
    }
}

void CppIncludeManager::writeUnoIncludes(CppFile& file)
{
    // Core UNO types - implemented: Any
    if (m_typeInfo.needsAny)
    {
        file.beginLine().append("#include <com/sun/star/uno/Any.hxx>").endLine();
        file.beginLine().append("#include <uno/data.h>").endLine();
        file.beginLine().append("#include <cppu/unotype.hxx>").endLine();
    }

    if (m_typeInfo.needsSequence)
    {
        file.beginLine().append("#include <com/sun/star/uno/Sequence.hxx>").endLine();
        file.beginLine().append("#include <uno/sequence2.h>").endLine();
        file.beginLine().append("#include <typelib/typedescription.h>").endLine();
    }

    if (m_typeInfo.needsPropertyValue)
    {
        // TODO: implement PropertyValue handling
        file.beginLine().append("// TODO: Add PropertyValue includes").endLine();
        file.beginLine().append("#include <com/sun/star/beans/PropertyValue.hpp>").endLine();
    }
}

void CppIncludeManager::writeInterfaceIncludes(CppFile& file, std::string_view interfaceName)
{
    // Include the interface being processed
    OString selfInclude = "<" + OString(interfaceName).replaceAll("."_ostr, "/"_ostr) + ".hpp>";
    file.beginLine().append("#include ").append(selfInclude).endLine();

    // Include headers for all interface types found in the analysis
    for (const auto& interfaceType : m_typeInfo.interfaceTypes)
    {
        // Don't include ourselves twice
        if (interfaceType != interfaceName)
        {
            OString includeFile = "<" + interfaceType.replaceAll("."_ostr, "/"_ostr) + ".hpp>";
            file.beginLine().append("#include ").append(includeFile).endLine();
        }
    }
}

OString CppIncludeManager::calculateRelativeIncludePath(const CppFile& file,
                                                        const OString& targetType,
                                                        const std::string& extension)
{
    namespace fs = std::filesystem;

    // Get the current file's directory
    fs::path currentFilePath = file.getPath();
    fs::path currentDir = currentFilePath.parent_path();

    // Convert target type name to path (dots to slashes)
    std::string targetTypeStr(targetType);
    std::replace(targetTypeStr.begin(), targetTypeStr.end(), '.', '/');

    // Find the base directory (cpp)
    fs::path baseDir = currentDir;
    while (baseDir.has_parent_path() && baseDir.filename() != "cpp")
    {
        baseDir = baseDir.parent_path();
    }

    // Create target file path
    fs::path target = (baseDir / (targetTypeStr + extension)).lexically_normal();

    // If they're in the same directory, just use the filename
    if (target.parent_path() == currentDir)
    {
        std::string result = target.filename().string();
        return OString("\"" + result + "\"");
    }

    // Otherwise calculate relative path from current directory to target file
    fs::path rel;
    try
    {
        rel = fs::relative(target, currentDir);
    }
    catch (const std::exception&)
    {
        // Fallback: use the path relative to base directory
        rel = fs::relative(target, baseDir);
    }

    std::string result = rel.generic_string();
    return OString("\"" + result + "\"");
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
